@extends('admin::layouts.admin')

@section('title', 'Ride Details')
@section('page-title', 'Ride Details')

@section('content')
@php
    $rideId = $ride['id'] ?? $ride['ride_id'] ?? 'Unknown';
    $rideStatus = $ride['status'] ?? 'unknown';
    $rideType = $ride['ride_type'] ?? 'standard';
    $paymentStatus = $ride['payment_status'] ?? 'pending';
    
    $createdAt = null;
    $createdDisplay = 'Unknown';
    if (!empty($ride['created_at'])) {
        try {
            $createdAt = \Carbon\Carbon::parse($ride['created_at']);
            $createdDisplay = $createdAt->format('M d, Y g:i A');
        } catch (\Exception $e) {
            // Handle date parsing error
        }
    }
    
    $fare = $ride['actual_fare'] ?? $ride['estimated_fare'] ?? 0;
    $distance = $ride['distance_km'] ?? 0;
    $duration = $ride['duration_minutes'] ?? 0;
@endphp

<!-- Header -->
<div class="flex justify-between items-center mb-6">
    <div>
        <h1 class="text-3xl font-bold text-primary">Ride Details</h1>
        <p class="text-gray-600 mt-1">Ride ID: {{ $rideId }}</p>
    </div>
    <div class="flex gap-3">
        <a href="{{ route('admin.rides.edit', $rideId) }}" 
           class="bg-success text-white px-4 py-2 rounded-lg hover:bg-green-600 transition-colors">
            <i class="fas fa-edit mr-2"></i>Edit Ride
        </a>
        <a href="{{ route('admin.rides.index') }}" 
           class="bg-gray-500 text-white px-4 py-2 rounded-lg hover:bg-gray-600 transition-colors">
            <i class="fas fa-arrow-left mr-2"></i>Back to Rides
        </a>
    </div>
</div>

@if(session('success'))
    <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
        <i class="fas fa-check-circle mr-2"></i>{{ session('success') }}
    </div>
@endif

@if(session('error'))
    <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
        <i class="fas fa-exclamation-circle mr-2"></i>{{ session('error') }}
    </div>
@endif

<!-- Ride Status and Quick Actions -->
<div class="bg-white rounded-lg shadow-sm border p-6 mb-6">
    <div class="flex justify-between items-center">
        <div class="flex items-center gap-4">
            <div>
                <h2 class="text-xl font-semibold mb-2">Current Status</h2>
                @switch($rideStatus)
                    @case('completed')
                        <span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-green-100 text-green-800">
                            <i class="fas fa-check-circle mr-2"></i>Completed
                        </span>
                        @break
                    @case('cancelled')
                        <span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-red-100 text-red-800">
                            <i class="fas fa-times-circle mr-2"></i>Cancelled
                        </span>
                        @break
                    @case('in_progress')
                        <span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-blue-100 text-blue-800">
                            <i class="fas fa-car mr-2"></i>In Progress
                        </span>
                        @break
                    @case('accepted')
                        <span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-yellow-100 text-yellow-800">
                            <i class="fas fa-clock mr-2"></i>Accepted
                        </span>
                        @break
                    @case('driver_arrived')
                        <span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-purple-100 text-purple-800">
                            <i class="fas fa-map-marker-alt mr-2"></i>Driver Arrived
                        </span>
                        @break
                    @default
                        <span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-gray-100 text-gray-800">
                            <i class="fas fa-question-circle mr-2"></i>{{ ucfirst($rideStatus) }}
                        </span>
                @endswitch
            </div>
            
            <div>
                <h3 class="text-lg font-medium mb-2">Payment Status</h3>
                @switch($paymentStatus)
                    @case('completed')
                        <span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-green-100 text-green-800">
                            <i class="fas fa-dollar-sign mr-2"></i>Paid
                        </span>
                        @break
                    @case('failed')
                        <span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-red-100 text-red-800">
                            <i class="fas fa-exclamation-triangle mr-2"></i>Failed
                        </span>
                        @break
                    @case('refunded')
                        <span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-blue-100 text-blue-800">
                            <i class="fas fa-undo mr-2"></i>Refunded
                        </span>
                        @break
                    @default
                        <span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-yellow-100 text-yellow-800">
                            <i class="fas fa-clock mr-2"></i>Pending
                        </span>
                @endswitch
            </div>
        </div>
        
        <!-- Quick Actions -->
        <div class="flex gap-2">
            @if(in_array($rideStatus, ['pending', 'requested', 'accepted']))
                <button onclick="updateRideStatus('{{ $rideId }}', 'in_progress')" 
                        class="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors">
                    <i class="fas fa-play mr-2"></i>Start Ride
                </button>
            @endif
            @if(in_array($rideStatus, ['in_progress', 'driver_arrived']))
                <button onclick="completeRide('{{ $rideId }}')" 
                        class="bg-green-600 text-white px-4 py-2 rounded-lg hover:bg-green-700 transition-colors">
                    <i class="fas fa-check mr-2"></i>Complete
                </button>
            @endif
            @if(!in_array($rideStatus, ['completed', 'cancelled']))
                <button onclick="cancelRide('{{ $rideId }}')" 
                        class="bg-red-600 text-white px-4 py-2 rounded-lg hover:bg-red-700 transition-colors">
                    <i class="fas fa-times mr-2"></i>Cancel
                </button>
            @endif
        </div>
    </div>
</div>

<!-- Main Content Grid -->
<div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
    
    <!-- Left Column - Main Details -->
    <div class="lg:col-span-2 space-y-6">
        
        <!-- Ride Information -->
        <div class="bg-white rounded-lg shadow-sm border">
            <div class="p-6 border-b">
                <h3 class="text-lg font-semibold">
                    <i class="fas fa-route mr-2 text-primary"></i>Ride Information
                </h3>
            </div>
            <div class="p-6">
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Ride ID</label>
                        <p class="text-gray-900 font-mono">{{ $rideId }}</p>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Ride Type</label>
                        <p class="text-gray-900">{{ ucfirst(str_replace('_', ' ', $rideType)) }}</p>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Created</label>
                        <p class="text-gray-900">{{ $createdDisplay }}</p>
                        @if($createdAt)
                            <p class="text-sm text-gray-500">{{ $createdAt->diffForHumans() }}</p>
                        @endif
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Payment Method</label>
                        <p class="text-gray-900">{{ ucfirst($ride['payment_method'] ?? 'Not specified') }}</p>
                    </div>
                </div>
                
                @if(!empty($ride['special_requests']))
                <div class="mt-4">
                    <label class="block text-sm font-medium text-gray-700 mb-1">Special Requests</label>
                    <p class="text-gray-900 bg-gray-50 p-3 rounded">{{ $ride['special_requests'] }}</p>
                </div>
                @endif
            </div>
        </div>
        
        <!-- Route Information -->
        <div class="bg-white rounded-lg shadow-sm border">
            <div class="p-6 border-b">
                <h3 class="text-lg font-semibold">
                    <i class="fas fa-map-marked-alt mr-2 text-primary"></i>Route Details
                </h3>
            </div>
            <div class="p-6">
                <div class="space-y-4">
                    <!-- Pickup Location -->
                    <div class="flex items-start">
                        <div class="flex-shrink-0 mt-1">
                            <div class="w-3 h-3 bg-green-500 rounded-full"></div>
                        </div>
                        <div class="ml-4 flex-1">
                            <h4 class="font-medium text-gray-900">Pickup Location</h4>
                            <p class="text-gray-600">{{ $ride['pickup_address'] ?? 'Address not available' }}</p>
                            @if(!empty($ride['pickup_latitude']) && !empty($ride['pickup_longitude']))
                                <p class="text-sm text-gray-500">{{ $ride['pickup_latitude'] }}, {{ $ride['pickup_longitude'] }}</p>
                            @endif
                        </div>
                    </div>
                    
                    <!-- Dropoff Location -->
                    <div class="flex items-start">
                        <div class="flex-shrink-0 mt-1">
                            <div class="w-3 h-3 bg-red-500 rounded-full"></div>
                        </div>
                        <div class="ml-4 flex-1">
                            <h4 class="font-medium text-gray-900">Dropoff Location</h4>
                            <p class="text-gray-600">{{ $ride['dropoff_address'] ?? 'Address not available' }}</p>
                            @if(!empty($ride['dropoff_latitude']) && !empty($ride['dropoff_longitude']))
                                <p class="text-sm text-gray-500">{{ $ride['dropoff_latitude'] }}, {{ $ride['dropoff_longitude'] }}</p>
                            @endif
                        </div>
                    </div>
                </div>
                
                <!-- Trip Metrics -->
                <div class="grid grid-cols-3 gap-4 mt-6 pt-6 border-t">
                    <div class="text-center">
                        <p class="text-2xl font-bold text-gray-900">{{ number_format($distance, 1) }}</p>
                        <p class="text-sm text-gray-500">Kilometers</p>
                    </div>
                    <div class="text-center">
                        <p class="text-2xl font-bold text-gray-900">{{ $duration }}</p>
                        <p class="text-sm text-gray-500">Minutes</p>
                    </div>
                    <div class="text-center">
                        <p class="text-2xl font-bold text-gray-900">${{ number_format($fare, 2) }}</p>
                        <p class="text-sm text-gray-500">{{ !empty($ride['actual_fare']) ? 'Actual' : 'Estimated' }} Fare</p>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Timeline -->
        <div class="bg-white rounded-lg shadow-sm border">
            <div class="p-6 border-b">
                <h3 class="text-lg font-semibold">
                    <i class="fas fa-clock mr-2 text-primary"></i>Ride Timeline
                </h3>
            </div>
            <div class="p-6">
                <div class="space-y-4">
                    @if(!empty($ride['requested_at']))
                    <div class="flex items-center">
                        <div class="w-3 h-3 bg-blue-500 rounded-full mr-4"></div>
                        <div>
                            <p class="font-medium">Ride Requested</p>
                            <p class="text-sm text-gray-500">{{ \Carbon\Carbon::parse($ride['requested_at'])->format('M d, Y g:i A') }}</p>
                        </div>
                    </div>
                    @endif
                    
                    @if(!empty($ride['accepted_at']))
                    <div class="flex items-center">
                        <div class="w-3 h-3 bg-yellow-500 rounded-full mr-4"></div>
                        <div>
                            <p class="font-medium">Driver Accepted</p>
                            <p class="text-sm text-gray-500">{{ \Carbon\Carbon::parse($ride['accepted_at'])->format('M d, Y g:i A') }}</p>
                        </div>
                    </div>
                    @endif
                    
                    @if(!empty($ride['driver_arrived_at']))
                    <div class="flex items-center">
                        <div class="w-3 h-3 bg-purple-500 rounded-full mr-4"></div>
                        <div>
                            <p class="font-medium">Driver Arrived</p>
                            <p class="text-sm text-gray-500">{{ \Carbon\Carbon::parse($ride['driver_arrived_at'])->format('M d, Y g:i A') }}</p>
                        </div>
                    </div>
                    @endif
                    
                    @if(!empty($ride['started_at']))
                    <div class="flex items-center">
                        <div class="w-3 h-3 bg-blue-600 rounded-full mr-4"></div>
                        <div>
                            <p class="font-medium">Ride Started</p>
                            <p class="text-sm text-gray-500">{{ \Carbon\Carbon::parse($ride['started_at'])->format('M d, Y g:i A') }}</p>
                        </div>
                    </div>
                    @endif
                    
                    @if(!empty($ride['completed_at']))
                    <div class="flex items-center">
                        <div class="w-3 h-3 bg-green-500 rounded-full mr-4"></div>
                        <div>
                            <p class="font-medium">Ride Completed</p>
                            <p class="text-sm text-gray-500">{{ \Carbon\Carbon::parse($ride['completed_at'])->format('M d, Y g:i A') }}</p>
                        </div>
                    </div>
                    @endif
                    
                    @if(!empty($ride['cancelled_at']))
                    <div class="flex items-center">
                        <div class="w-3 h-3 bg-red-500 rounded-full mr-4"></div>
                        <div>
                            <p class="font-medium">Ride Cancelled</p>
                            <p class="text-sm text-gray-500">{{ \Carbon\Carbon::parse($ride['cancelled_at'])->format('M d, Y g:i A') }}</p>
                            @if(!empty($ride['cancellation_reason']))
                                <p class="text-sm text-red-600 mt-1">Reason: {{ $ride['cancellation_reason'] }}</p>
                            @endif
                        </div>
                    </div>
                    @endif
                </div>
            </div>
        </div>
        
        <!-- Ratings and Feedback -->
        @if(($ride['driver_rating'] ?? 0) > 0 || ($ride['passenger_rating'] ?? 0) > 0 || !empty($ride['driver_feedback']) || !empty($ride['passenger_feedback']))
        <div class="bg-white rounded-lg shadow-sm border">
            <div class="p-6 border-b">
                <h3 class="text-lg font-semibold">
                    <i class="fas fa-star mr-2 text-primary"></i>Ratings & Feedback
                </h3>
            </div>
            <div class="p-6">
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    @if(($ride['driver_rating'] ?? 0) > 0)
                    <div>
                        <h4 class="font-medium text-gray-900 mb-2">Driver Rating</h4>
                        <div class="flex items-center mb-2">
                            @for($i = 1; $i <= 5; $i++)
                                <i class="fas fa-star {{ $i <= ($ride['driver_rating'] ?? 0) ? 'text-yellow-400' : 'text-gray-300' }} mr-1"></i>
                            @endfor
                            <span class="ml-2 text-gray-600">{{ $ride['driver_rating'] }}/5</span>
                        </div>
                        @if(!empty($ride['driver_feedback']))
                            <p class="text-gray-700 bg-gray-50 p-3 rounded text-sm">{{ $ride['driver_feedback'] }}</p>
                        @endif
                    </div>
                    @endif
                    
                    @if(($ride['passenger_rating'] ?? 0) > 0)
                    <div>
                        <h4 class="font-medium text-gray-900 mb-2">Passenger Rating</h4>
                        <div class="flex items-center mb-2">
                            @for($i = 1; $i <= 5; $i++)
                                <i class="fas fa-star {{ $i <= ($ride['passenger_rating'] ?? 0) ? 'text-yellow-400' : 'text-gray-300' }} mr-1"></i>
                            @endfor
                            <span class="ml-2 text-gray-600">{{ $ride['passenger_rating'] }}/5</span>
                        </div>
                        @if(!empty($ride['passenger_feedback']))
                            <p class="text-gray-700 bg-gray-50 p-3 rounded text-sm">{{ $ride['passenger_feedback'] }}</p>
                        @endif
                    </div>
                    @endif
                </div>
            </div>
        </div>
        @endif
    </div>
    
    <!-- Right Column - People and Vehicle Info -->
    <div class="space-y-6">
        
        <!-- Driver Information -->
        @if($driver)
        <div class="bg-white rounded-lg shadow-sm border">
            <div class="p-6 border-b">
                <h3 class="text-lg font-semibold">
                    <i class="fas fa-user-tie mr-2 text-primary"></i>Driver Information
                </h3>
            </div>
            <div class="p-6">
                <div class="flex items-center mb-4">
                    @if(!empty($driver['profile_image']))
                        <img src="{{ $driver['profile_image'] }}" alt="{{ $driver['name'] }}" 
                             class="w-12 h-12 rounded-full object-cover mr-4">
                    @else
                        <div class="w-12 h-12 bg-gray-300 rounded-full flex items-center justify-center mr-4">
                            <i class="fas fa-user text-gray-600"></i>
                        </div>
                    @endif
                    <div>
                        <h4 class="font-semibold text-gray-900">{{ $driver['name'] ?? 'Unknown Driver' }}</h4>
                        <p class="text-gray-600">{{ $driver['email'] ?? 'Email not available' }}</p>
                    </div>
                </div>
                
                <div class="space-y-3">
                    @if(!empty($driver['phone']))
                    <div class="flex justify-between">
                        <span class="text-gray-600">Phone:</span>
                        <span class="text-gray-900">{{ $driver['phone'] }}</span>
                    </div>
                    @endif
                    
                    @if(!empty($driver['license_number']))
                    <div class="flex justify-between">
                        <span class="text-gray-600">License:</span>
                        <span class="text-gray-900">{{ $driver['license_number'] }}</span>
                    </div>
                    @endif
                    
                    @if(!empty($driver['rating']))
                    <div class="flex justify-between items-center">
                        <span class="text-gray-600">Rating:</span>
                        <div class="flex items-center">
                            <i class="fas fa-star text-yellow-400 mr-1"></i>
                            <span class="text-gray-900">{{ number_format($driver['rating'], 1) }}</span>
                        </div>
                    </div>
                    @endif
                </div>
                
                <div class="mt-4 pt-4 border-t">
                    <a href="{{ route('admin.drivers.show', $driver['firebase_uid']) }}" 
                       class="text-primary hover:text-blue-700 text-sm">
                        <i class="fas fa-external-link-alt mr-1"></i>View Driver Profile
                    </a>
                </div>
            </div>
        </div>
        @endif
        
        <!-- Passenger Information -->
        <div class="bg-white rounded-lg shadow-sm border">
            <div class="p-6 border-b">
                <h3 class="text-lg font-semibold">
                    <i class="fas fa-user mr-2 text-primary"></i>Passenger Information
                </h3>
            </div>
            <div class="p-6">
                <div class="space-y-3">
                    <div class="flex justify-between">
                        <span class="text-gray-600">Name:</span>
                        <span class="text-gray-900">{{ $ride['passenger_name'] ?? 'Unknown Passenger' }}</span>
                    </div>
                    
                    @if(!empty($ride['passenger_phone']))
                    <div class="flex justify-between">
                        <span class="text-gray-600">Phone:</span>
                        <span class="text-gray-900">{{ $ride['passenger_phone'] }}</span>
                    </div>
                    @endif
                    
                    @if(!empty($ride['passenger_email']))
                    <div class="flex justify-between">
                        <span class="text-gray-600">Email:</span>
                        <span class="text-gray-900">{{ $ride['passenger_email'] }}</span>
                    </div>
                    @endif
                </div>
            </div>
        </div>
        
        <!-- Vehicle Information -->
        @if($vehicle)
        <div class="bg-white rounded-lg shadow-sm border">
            <div class="p-6 border-b">
                <h3 class="text-lg font-semibold">
                    <i class="fas fa-car mr-2 text-primary"></i>Vehicle Information
                </h3>
            </div>
            <div class="p-6">
                <div class="space-y-3">
                    <div class="flex justify-between">
                        <span class="text-gray-600">Make & Model:</span>
                        <span class="text-gray-900">{{ $vehicle['make'] ?? '' }} {{ $vehicle['model'] ?? '' }}</span>
                    </div>
                    
                    @if(!empty($vehicle['year']))
                    <div class="flex justify-between">
                        <span class="text-gray-600">Year:</span>
                        <span class="text-gray-900">{{ $vehicle['year'] }}</span>
                    </div>
                    @endif
                    
                    @if(!empty($vehicle['license_plate']))
                    <div class="flex justify-between">
                        <span class="text-gray-600">License Plate:</span>
                        <span class="text-gray-900 font-mono">{{ $vehicle['license_plate'] }}</span>
                    </div>
                    @endif
                    
                    @if(!empty($vehicle['color']))
                    <div class="flex justify-between">
                        <span class="text-gray-600">Color:</span>
                        <span class="text-gray-900">{{ $vehicle['color'] }}</span>
                    </div>
                    @endif
                </div>
                
                <div class="mt-4 pt-4 border-t">
                    <a href="{{ route('admin.vehicles.show', $vehicle['id']) }}" 
                       class="text-primary hover:text-blue-700 text-sm">
                        <i class="fas fa-external-link-alt mr-1"></i>View Vehicle Details
                    </a>
                </div>
            </div>
        </div>
        @endif
        
        <!-- Financial Details -->
        <div class="bg-white rounded-lg shadow-sm border">
            <div class="p-6 border-b">
                <h3 class="text-lg font-semibold">
                    <i class="fas fa-money-bill mr-2 text-primary"></i>Financial Details
                </h3>
            </div>
            <div class="p-6">
                <div class="space-y-3">
                    @if(!empty($ride['estimated_fare']))
                    <div class="flex justify-between">
                        <span class="text-gray-600">Estimated Fare:</span>
                        <span class="text-gray-900">${{ number_format($ride['estimated_fare'], 2) }}</span>
                    </div>
                    @endif
                    
                    @if(!empty($ride['actual_fare']))
                    <div class="flex justify-between">
                        <span class="text-gray-600">Actual Fare:</span>
                        <span class="text-gray-900 font-semibold">${{ number_format($ride['actual_fare'], 2) }}</span>
                    </div>
                    @endif
                    
                    @if(!empty($ride['driver_earnings']))
                    <div class="flex justify-between">
                        <span class="text-gray-600">Driver Earnings:</span>
                        <span class="text-gray-900">${{ number_format($ride['driver_earnings'], 2) }}</span>
                    </div>
                    @endif
                    
                    @if(!empty($ride['platform_fee']))
                    <div class="flex justify-between">
                        <span class="text-gray-600">Platform Fee:</span>
                        <span class="text-gray-900">${{ number_format($ride['platform_fee'], 2) }}</span>
                    </div>
                    @endif
                    
                    @if(!empty($ride['tip_amount']))
                    <div class="flex justify-between">
                        <span class="text-gray-600">Tip:</span>
                        <span class="text-gray-900">${{ number_format($ride['tip_amount'], 2) }}</span>
                    </div>
                    @endif
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
    function showNotification(message, type = 'info') {
        const alertClass = {
            'success': 'bg-green-100 border-green-400 text-green-700',
            'error': 'bg-red-100 border-red-400 text-red-700',
            'info': 'bg-blue-100 border-blue-400 text-blue-700',
            'warning': 'bg-yellow-100 border-yellow-400 text-yellow-700'
        }[type] || 'bg-gray-100 border-gray-400 text-gray-700';

        const notification = document.createElement('div');
        notification.className = `${alertClass} px-4 py-3 rounded mb-4 fixed top-4 right-4 z-50 min-w-80 shadow-lg`;
        notification.innerHTML = `
            <div class="flex justify-between items-center">
                <span>${message}</span>
                <button onclick="this.parentElement.parentElement.remove()" class="ml-4 text-lg">&times;</button>
            </div>
        `;
        
        document.body.appendChild(notification);
        setTimeout(() => {
            if (notification.parentNode) {
                notification.remove();
            }
        }, 5000);
    }

    async function updateRideStatus(rideId, status) {
        const statusText = status.replace('_', ' ');
        if (!confirm(`Are you sure you want to ${statusText} this ride?`)) {
            return;
        }
        
        try {
            const response = await fetch(`{{ route('admin.rides.update-status', '') }}/${rideId}`, {
                method: 'PATCH',
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                    'Content-Type': 'application/json',
                    'Accept': 'application/json'
                },
                body: JSON.stringify({ status: status })
            });
            
            const result = await response.json();
            
            if (response.ok && result.success) {
                showNotification(`Ride ${statusText} successfully`, 'success');
                setTimeout(() => window.location.reload(), 1000);
            } else {
                showNotification(`Failed to ${statusText} ride: ` + (result.message || 'Unknown error'), 'error');
            }
        } catch (error) {
            console.error('Update status error:', error);
            showNotification(`Error ${statusText} ride: Connection failed`, 'error');
        }
    }

    async function completeRide(rideId) {
        if (!confirm('Are you sure you want to complete this ride?')) {
            return;
        }
        
        // Optionally collect completion data
        const actualFare = prompt('Enter actual fare (optional):');
        const completionData = {};
        
        if (actualFare && !isNaN(parseFloat(actualFare))) {
            completionData.actual_fare = parseFloat(actualFare);
        }
        
        try {
            const response = await fetch(`{{ route('admin.rides.complete', '') }}/${rideId}`, {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                    'Content-Type': 'application/json',
                    'Accept': 'application/json'
                },
                body: JSON.stringify(completionData)
            });
            
            const result = await response.json();
            
            if (response.ok && result.success) {
                showNotification('Ride completed successfully', 'success');
                setTimeout(() => window.location.reload(), 1000);
            } else {
                showNotification('Failed to complete ride: ' + (result.message || 'Unknown error'), 'error');
            }
        } catch (error) {
            console.error('Complete ride error:', error);
            showNotification('Error completing ride: Connection failed', 'error');
        }
    }

    async function cancelRide(rideId) {
        const reason = prompt('Please provide a cancellation reason:');
        if (!reason) return;
        
        try {
            const response = await fetch(`{{ route('admin.rides.cancel', '') }}/${rideId}`, {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                    'Content-Type': 'application/json',
                    'Accept': 'application/json'
                },
                body: JSON.stringify({ cancellation_reason: reason })
            });
            
            const result = await response.json();
            
            if (response.ok && result.success) {
                showNotification('Ride cancelled successfully', 'success');
                setTimeout(() => window.location.reload(), 1000);
            } else {
                showNotification('Failed to cancel ride: ' + (result.message || 'Unknown error'), 'error');
            }
        } catch (error) {
            console.error('Cancel ride error:', error);
            showNotification('Error cancelling ride: Connection failed', 'error');
        }
    }
    
    document.addEventListener('DOMContentLoaded', function() {
        console.log('Ride details page initialized');
    });
</script>
@endpush